// Perch Browser Observer - Capture
// Console log capture, network request tracking, URL updates, message handling

import {
    encodeLogDatagram,
    encodeNetworkDatagram,
    encodeUrlUpdateDatagram,
    logLevelToNumber,
} from "./codec.js";

let networkRequests = new Map();

// Send current tab URL via datagram
export async function sendCurrentUrl(sendDatagram) {
    try {
        const [tab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (tab) {
            sendDatagram(encodeUrlUpdateDatagram(tab.url, tab.title));
        }
    } catch (e) {
        console.error("[Perch] Failed to get current URL:", e);
    }
}

// Set up all browser event listeners for capture
// ctx: { getConfig, sendDatagram, isConnected, shouldSendLog, matchesPinnedTab,
//         saveConfig, disconnect, connect, getStatus, updateBadge, resetReconnect }
export function setupCapture(ctx) {
    // Track tab URL changes
    browser.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
        if (changeInfo.url && tab.active) {
            ctx.sendDatagram(
                encodeUrlUpdateDatagram(changeInfo.url, tab.title || ""),
            );
        }
    });

    // Track active tab changes
    browser.tabs.onActivated.addListener(async (activeInfo) => {
        try {
            const tab = await browser.tabs.get(activeInfo.tabId);
            ctx.sendDatagram(encodeUrlUpdateDatagram(tab.url, tab.title));
        } catch {}
        // Update badge to reflect whether active tab is pinned
        ctx.updateBadge();
    });

    // Track network requests - start
    // Note: Firefox uses documentUrl, Chrome uses initiator
    browser.webRequest.onBeforeRequest.addListener(
        (details) => {
            if (!ctx.getConfig().enabled) return;
            const originUrl =
                details.documentUrl ??
                details.initiator ??
                details.originUrl ??
                details.url;
            if (!ctx.matchesPinnedTab(details.tabId, originUrl)) return;
            networkRequests.set(details.requestId, {
                startTime: Date.now(),
                method: details.method,
                url: details.url,
            });
        },
        { urls: ["<all_urls>"] },
    );

    // Track network requests - completion
    browser.webRequest.onCompleted.addListener(
        (details) => {
            if (!ctx.getConfig().enabled) return;

            const request = networkRequests.get(details.requestId);
            networkRequests.delete(details.requestId);

            if (!request) return;

            const duration = Date.now() - request.startTime;
            ctx.sendDatagram(
                encodeNetworkDatagram(
                    request.method,
                    request.url,
                    details.statusCode,
                    duration,
                    Date.now(),
                ),
            );
        },
        { urls: ["<all_urls>"] },
    );

    // Track network request errors
    browser.webRequest.onErrorOccurred.addListener(
        (details) => {
            if (!ctx.getConfig().enabled) return;

            const request = networkRequests.get(details.requestId);
            networkRequests.delete(details.requestId);

            if (!request) return;

            const duration = Date.now() - request.startTime;
            ctx.sendDatagram(
                encodeNetworkDatagram(
                    request.method,
                    request.url,
                    0,
                    duration,
                    Date.now(),
                ),
            );
        },
        { urls: ["<all_urls>"] },
    );

    // Clean up stale network requests
    setInterval(() => {
        const fiveMinutesAgo = Date.now() - 5 * 60 * 1000;
        for (const [requestId, request] of networkRequests) {
            if (request.startTime < fiveMinutesAgo) {
                networkRequests.delete(requestId);
            }
        }
    }, 60000);

    // Handle messages from popup and content scripts
    console.log("[Perch] Registering message listener");
    browser.runtime.onMessage.addListener((message, sender, sendResponse) => {
        console.log("[Perch] Received message:", message.type);
        // Content script: console logs
        if (message.type === "log") {
            const tabId = sender.tab?.id;
            if (
                ctx.getConfig().enabled &&
                ctx.shouldSendLog(message.level) &&
                ctx.matchesPinnedTab(tabId, message.url)
            ) {
                const level = logLevelToNumber(message.level);
                ctx.sendDatagram(
                    encodeLogDatagram(
                        level,
                        message.message,
                        message.url,
                        message.timestamp,
                        message.source_file,
                        message.line_number,
                    ),
                );
            }
            return false;
        }

        // Popup: get status
        if (message.type === "getStatus") {
            const response = ctx.getStatus();
            console.log("[Perch] Sending getStatus response:", response);
            sendResponse(response);
            return true;
        }

        if (message.type === "saveConfig") {
            ctx.saveConfig(message.config).then(() => {
                ctx.updateBadge();
                sendResponse({ success: true });
            });
            return true;
        }

        if (message.type === "reconnect") {
            ctx.disconnect();
            ctx.resetReconnect();
            if (ctx.getConfig().enabled) {
                ctx.connect();
            }
            sendResponse({ success: true });
            return true;
        }
    });
}
